#include <SDL.h>
#include <SDL_video.h>
#include <SDL_surface.h>
#include <iostream>

#include <CLI/CLI.hpp>
#include <spdlog/spdlog.h>
#include "render.hpp"


// Usage: ./view resolution

int main(int argc, char** argv)
{
  std::string filename;
  std::string method = "base";
  int width = 0;
  int height = 1080;
  int niter = 100;

  
  CLI::App app{"mandel"};
  app.add_option("-o", filename, "Output BMP image");
  app.add_option("-n", niter, "number of iteration");
  app.add_option("height", height, "height of the output image");
  app.add_option("width", width, "width of the output image");
  app.add_set("-m", method, {"base", "mt", "optim", "simd", "final"}, "Method to run:  {base, mt, optim, simd, final}");

  CLI11_PARSE(app, argc, argv);

  if (SDL_Init(SDL_INIT_VIDEO) != 0)
  {
    spdlog::error("{}:{} Unable to init SDL ({})", __FILE__, __LINE__, SDL_GetError());
    return 1;
  }

  height = std::clamp(height, 360, 3200);
  width = (width > 0) ? width : height * 16 / 9;


  // Rendering
  auto img = SDL_CreateRGBSurfaceWithFormat(0, width, height, 24, SDL_PIXELFORMAT_RGB24);
  SDL_LockSurface(img);

  spdlog::info("Running method: '{}'", method);
  if (method == "base")
    render_base(static_cast<std::byte*>(img->pixels), width, height, img->pitch);
  else
    spdlog::error("Method not yet implemented");
  SDL_UnlockSurface(img);


  if (!filename.empty())
  {
    SDL_SaveBMP(img, filename.c_str());
  }
  else
  {
    // Create window
    auto window = SDL_CreateWindow("Render", SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, width, height,
                                   SDL_WINDOW_SHOWN | SDL_WINDOW_BORDERLESS);
    if (window == nullptr)
    {
      spdlog::error("{}:{} Unable to create window ({})", __FILE__, __LINE__, SDL_GetError());
      return 1;
    }
    auto surface = SDL_GetWindowSurface(window);

    //Update the surface
    if (SDL_BlitSurface(img, nullptr, surface, nullptr) != 0)
    {
      spdlog::error("{}:{} Cannot blit surface.", __FILE__, __LINE__);
    }
    SDL_UpdateWindowSurface(window);
    SDL_Delay(5000);
    SDL_DestroyWindow(window);
  }

  // Quit
  SDL_Quit();
}
